local ADDON_NAME = "WZRaidFinder"
local WZRaidFinder = LibStub("AceAddon-3.0"):NewAddon(ADDON_NAME, "AceConsole-3.0", "AceEvent-3.0", "AceTimer-3.0")
local AceGUI = LibStub("AceGUI-3.0")

if not _G["WZRaidFinder"] then
	_G["WZRaidFinder"] = WZRaidFinder
end

WZRaidFinder.VERSION = "2.1.0"
WZRaidFinder.CHAT_PREFIX = "|cFF16C3F2[WZ Raid Finder]|r"

WZRaidFinder.raidList = {}
WZRaidFinder.availableInstances = {}
WZRaidFinder.myRaidInfo = nil
WZRaidFinder.isReceivingData = false
WZRaidFinder.pendingRaidData = {}
WZRaidFinder.dataType = nil
WZRaidFinder.serverCommandsAvailable = nil
WZRaidFinder.commandCheckAttempted = false

local defaults = {
	profile = {
		minimap = {
			hide = false,
		},
		chat = {
			showMessages = false,
		},
		filters = {
			minGearScore = 0,
			instance = nil,
			showLegacyInstances = false,
			showCrossFaction = true,
			showFullRaids = false,
			showAlliance = true,
			showHorde = true,
		},
	}
}

function WZRaidFinder:OnInitialize()
	self.db = LibStub("AceDB-3.0"):New("WZRaidFinderDB", defaults, true)
	
	self.db.profile.filters.minGearScore = 0
	self.db.profile.filters.instance = nil
	self.db.profile.filters.showCrossFaction = true
	self.db.profile.filters.showFullRaids = false
	self.db.profile.filters.showAlliance = true
	self.db.profile.filters.showHorde = true
	
	self:RegisterChatCommand("wzrf", "ChatCommand")
	
	self:Print("WZ Raid Finder v" .. self.VERSION .. " loaded! Type /wzrf to open.")
end

function WZRaidFinder:OnEnable()
	self:RegisterEvent("CHAT_MSG_SYSTEM")
	
	ChatFrame_AddMessageEventFilter("CHAT_MSG_SYSTEM", function(self, event, msg, ...)
		if msg and msg:find("^WZRF:") then
			return true
		end
		return false
	end)
	
	self:ScheduleTimer(function()
		self:MinimapButton_Initialize()
	end, 0.5)
	
	self:ScheduleTimer("RequestInstanceList", 0.5)
end

function WZRaidFinder:OnDisable()
	self:CancelAllTimers()
end

function WZRaidFinder:ChatCommand(input)
	if not input or input:trim() == "" then
		self:OpenMainGUI()
	elseif input == "list" then
		self:RequestRaidList()
	elseif input == "minimap" then
		self.db.profile.minimap.hide = not self.db.profile.minimap.hide
		self:MinimapButton_Toggle()
	else
		self:OpenMainGUI()
	end
end

function WZRaidFinder:CHAT_MSG_SYSTEM(event, msg)
	if msg and msg:find("^WZRF:") then
		self.serverCommandsAvailable = true
		self:ParseServerMessage(msg)
		return true
	elseif msg and (msg:find("Unknown command") or msg:find("No such command")) then
		if self.commandCheckAttempted and not self.serverCommandsAvailable then
			self.serverCommandsAvailable = false
			self:Print("|cFFFF0000Server commands not available. Please ensure the WZ Raid Finder server addon is installed.|r")
		end
	end
end

function WZRaidFinder:ParseServerMessage(msg)
	if msg == "WZRF:BEGIN" then
		self.isReceivingData = true
		self.dataType = "RAIDLIST"
		self.pendingRaidData = {}
		return
	elseif msg == "WZRF:MYRAID:BEGIN" then
		self.isReceivingData = true
		self.dataType = "MYRAID"
		self.myRaidInfo = nil
		return
	elseif msg == "WZRF:INSTANCES:BEGIN" then
		self.isReceivingData = true
		self.dataType = "INSTANCES"
		self.availableInstances = {}
		return
	elseif msg == "WZRF:DETAILS:BEGIN" then
		self.isReceivingData = true
		self.dataType = "DETAILS"
		self.raidDetails = {members = {}}
		return
	end
	
	if msg:match("^WZRF:END%|(%d+)") then
		local count = tonumber(msg:match("^WZRF:END%|(%d+)"))
		self:OnRaidListReceived(count)
		return
	elseif msg:match("^WZRF:MYRAID:END%|(%d+)") then
		local count = tonumber(msg:match("^WZRF:MYRAID:END%|(%d+)"))
		self:OnMyRaidReceived(count)
		return
	elseif msg == "WZRF:INSTANCES:END" then
		self:OnInstancesReceived()
		return
	elseif msg:match("^WZRF:DETAILS:END%|(%d+)") then
		local count = tonumber(msg:match("^WZRF:DETAILS:END%|(%d+)"))
		self:OnDetailsReceived(count)
		return
	elseif msg:match("^WZRF:ERROR%|(.+)") then
		local error = msg:match("^WZRF:ERROR%|(.+)")
		self:Print("|cffff0000Error: " .. error)
		self.isReceivingData = false
		return
	end
	
	if self.isReceivingData and self.dataType == "RAIDLIST" then
		local parts = {strsplit("|", msg)}
		if parts[1] == "WZRF:RAID" and #parts >= 11 then
			local current, max = parts[9]:match("(%d+)/(%d+)")
			table.insert(self.pendingRaidData, {
				id = tonumber(parts[2]),
				map = parts[3],
				mode = parts[4],
				leader = parts[5],
				leaderClass = parts[6],
				hasCD = tonumber(parts[7]) == 1,
				faction = tonumber(parts[8]),
				current = tonumber(current),
				max = tonumber(max),
				minGS = tonumber(parts[10]),
				isCross = tonumber(parts[11]) == 1,
			})
		end
	elseif self.isReceivingData and self.dataType == "MYRAID" then
		local parts = {strsplit("|", msg)}
		if parts[1] == "WZRF:MYRAID" and #parts >= 7 then
			local current, max = parts[5]:match("(%d+)/(%d+)")
			self.myRaidInfo = {
				id = tonumber(parts[2]),
				map = parts[3],
				mode = parts[4],
				current = tonumber(current),
				max = tonumber(max),
				minGS = tonumber(parts[6]),
				isCross = tonumber(parts[7]) == 1,
			}
		end
	elseif self.isReceivingData and self.dataType == "INSTANCES" then
		local parts = {strsplit("|", msg)}
		if parts[1] == "WZRF:INST" and #parts >= 6 then
			table.insert(self.availableInstances, {
				acronym = parts[2],
				name = parts[3],
				modes = tonumber(parts[4]),
				requiredLevel = tonumber(parts[5]),
				expansion = tonumber(parts[6]),
			})
		end
	elseif self.isReceivingData and self.dataType == "DETAILS" then
		local parts = {strsplit("|", msg)}
		if parts[1] == "WZRF:DETAILS" and #parts >= 8 then
			self.raidDetails.id = tonumber(parts[2])
			self.raidDetails.map = parts[3]
			self.raidDetails.mode = parts[4]
			self.raidDetails.leader = parts[5]
			self.raidDetails.minGS = tonumber(parts[6])
			self.raidDetails.isCross = tonumber(parts[7]) == 1
			self.raidDetails.maxMembers = tonumber(parts[8])
		elseif parts[1] == "WZRF:MEMBER" and #parts >= 6 then
			table.insert(self.raidDetails.members, {
				name = parts[2],
				class = tonumber(parts[3]),
				level = tonumber(parts[4]),
				gearscore = tonumber(parts[5]),
				role = tonumber(parts[6]),
			})
		end
	end
end

function WZRaidFinder:OnRaidListReceived(count)
	self.isReceivingData = false
	self.raidList = self.pendingRaidData
	self.pendingRaidData = {}
	self.dataType = nil
	self.lastRaidListUpdate = time()
	
	if self.GUI and self.GUI.mainFrame and self.GUI.mainFrame:IsShown() then
		if self.GUI.currentTab == "list" and self.GUI.raidListContainer then
			self.GUI.raidListContainer:ReleaseChildren()
			self:GUI_DrawRaidList(self.GUI.raidListContainer)
		end
	end
end

function WZRaidFinder:OnMyRaidReceived(count)
	self.isReceivingData = false
	self.dataType = nil
	
	if self.GUI and self.GUI.mainFrame and self.GUI.mainFrame:IsShown() then
		if self.GUI.currentTab == "myraid" and self.GUI.myRaidContainer then
			self.GUI.myRaidContainer:ReleaseChildren()
			self:GUI_DrawMyRaid(self.GUI.myRaidContainer)
		end
	end
end

function WZRaidFinder:OnInstancesReceived()
	self.isReceivingData = false
	self.dataType = nil
	self:Print(string.format("Loaded %d instance(s)", #self.availableInstances))
end

function WZRaidFinder:OnDetailsReceived(count)
	self.isReceivingData = false
	self.dataType = nil
	
	if count and count > 0 and self.raidDetails then
		self:GUI_ShowRaidDetails(self.raidDetails)
	end
end

function WZRaidFinder:SendCommand(cmd)
	if not cmd then return end
	SendChatMessage(cmd, "SAY")
end

function WZRaidFinder:RequestRaidList(instanceFilter)
	if self.serverCommandsAvailable == false then
		self:Print("|cFFFF0000Server commands not available.|r")
		return
	end
	
	local cmd = ".wzrf getlist"
	if instanceFilter then
		cmd = cmd .. " " .. instanceFilter
	end
	self:SendCommand(cmd)
end

function WZRaidFinder:RequestMyRaid()
	if self.serverCommandsAvailable == false then
		self:Print("|cFFFF0000Server commands not available.|r")
		return
	end
	
	self:SendCommand(".wzrf getmyraid")
end

function WZRaidFinder:RequestInstanceList()
	if self.serverCommandsAvailable == false then
		return
	end
	
	self.commandCheckAttempted = true
	self:SendCommand(".wzrf getinstances")
	
	if self.serverCommandsAvailable == nil then
		self:ScheduleTimer(function()
			if self.serverCommandsAvailable == nil then
				self.serverCommandsAvailable = false
				self:Print("|cFFFF0000Server commands not responding. WZ Raid Finder server addon may not be installed.|r")
			end
		end, 3)
	end
end

function WZRaidFinder:RequestRaidDetails(raidId)
	if not raidId then
		self:Print("Invalid raid ID")
		return
	end
	
	if self.serverCommandsAvailable == false then
		self:Print("|cFFFF0000Server commands not available.|r")
		return
	end
	
	self:SendCommand(".wzrf getdetails " .. raidId)
end

function WZRaidFinder:Print(msg)
	if self.db.profile.chat.showMessages then
		DEFAULT_CHAT_FRAME:AddMessage(self.CHAT_PREFIX .. " " .. msg)
	end
end

function WZRaidFinder:GetFactionColor(faction)
	if faction == 0 then
		return "|cFF0080FFAlliance|r"
	else
		return "|cFFFF0000Horde|r"
	end
end

function WZRaidFinder:GetModeColor(mode)
	if mode:find("H") then
		return "|cFFFF0000"
	else
		return "|cFF00FF00"
	end
end

