local WZRaidFinder = _G.WZRaidFinder
local LDB = LibStub("LibDataBroker-1.1", true)
local LibDBIcon = LibStub("LibDBIcon-1.0", true)

WZRaidFinder.MinimapButton = {}

function WZRaidFinder:MinimapButton_Initialize()
	if LDB and LibDBIcon then
		self:MinimapButton_InitializeWithLibs()
		return
	end
	
	self:MinimapButton_CreateSimple()
end

function WZRaidFinder:MinimapButton_InitializeWithLibs()
	if not LDB or not LibDBIcon then
		return
	end
	
	local dataObj = LDB:NewDataObject("WZRaidFinder", {
		type = "launcher",
		text = "WZ Raid Finder",
		icon = "Interface\\Icons\\Achievement_Boss_LichKing",
		OnClick = function(frame, button)
			WZRaidFinder:OpenMainGUI()
		end,
		OnTooltipShow = function(tooltip)
			if not tooltip then return end
			tooltip:SetText("|cFF16C3F2WZ Raid Finder|r")
			tooltip:AddLine("Version: " .. WZRaidFinder.VERSION, 1, 1, 1)
			tooltip:AddLine(" ")
			tooltip:AddLine("|cFFFFFFFFClick to open|r", 0, 1, 0)
			
			local count = #WZRaidFinder.raidList
			if count > 0 then
				tooltip:AddLine(" ")
				tooltip:AddLine(string.format("|cFF00FF00%d raid(s) available|r", count), 1, 1, 1)
			end
		end,
	})
	
	LibDBIcon:Register("WZRaidFinder", dataObj, self.db.profile.minimap)
	
	self.MinimapButton.dataObj = dataObj
	self.MinimapButton.icon = LibDBIcon:GetMinimapButton("WZRaidFinder")
end

function WZRaidFinder:MinimapButton_Toggle()
	if LibDBIcon and self.MinimapButton.dataObj then
		if self.db.profile.minimap.hide then
			LibDBIcon:Hide("WZRaidFinder")
		else
			LibDBIcon:Show("WZRaidFinder")
		end
		return
	end
	
	if self.MinimapButton.simpleButton then
		if self.db.profile.minimap.hide then
			self.MinimapButton.simpleButton:Hide()
		else
			self.MinimapButton.simpleButton:Show()
		end
	end
end

function WZRaidFinder:MinimapButton_Flash()
	if not self.MinimapButton.icon then return end
	
	local icon = self.MinimapButton.icon
	local flashCount = 0
	local maxFlashes = 6
	
	local function flash()
		flashCount = flashCount + 1
		if flashCount > maxFlashes then return end
		
		if flashCount % 2 == 0 then
			icon:SetAlpha(0.3)
		else
			icon:SetAlpha(1.0)
		end
		
		WZRaidFinder:ScheduleTimer(flash, 0.3)
	end
	
	flash()
end

function WZRaidFinder:MinimapButton_UpdateCount()
	if not self.MinimapButton.icon then return end
	
	local count = #self.raidList
	
	if self.MinimapButton.dataObj then
		self.MinimapButton.dataObj.text = string.format("WZ Raid Finder (%d)", count)
	end
	
	if not self.MinimapButton.countBadge then
		local badge = self.MinimapButton.icon:CreateFontString(nil, "OVERLAY", "GameFontNormalSmall")
		badge:SetPoint("TOPRIGHT", self.MinimapButton.icon, "TOPRIGHT", 2, 2)
		badge:SetTextColor(1, 1, 0)
		self.MinimapButton.countBadge = badge
	end
	
	if count > 0 then
		self.MinimapButton.countBadge:SetText(tostring(count))
		self.MinimapButton.countBadge:Show()
	else
		self.MinimapButton.countBadge:Hide()
	end
end

function WZRaidFinder:ShowMinimapMenu(frame)
	local menu = {
		{
			text = "WZ Raid Finder",
			isTitle = true,
			notCheckable = true,
		},
		{
			text = "Open Raid List",
			func = function()
				self:OpenMainGUI()
			end,
			notCheckable = true,
		},
		{
			text = "Refresh List",
			func = function()
				self:RefreshRaidList()
			end,
			notCheckable = true,
		},
		{
			text = "Create Raid",
			func = function()
				self:OpenCreateGUI()
			end,
			notCheckable = true,
		},
		{
			text = " ",
			isTitle = true,
			notCheckable = true,
		},
		{
			text = "Available Raids",
			func = function()
				self:SendServerCommand(".raid raids")
			end,
			notCheckable = true,
		},
		{
			text = "My Raid Info",
			func = function()
				self:SendServerCommand(".raid info")
			end,
			notCheckable = true,
		},
		{
			text = " ",
			isTitle = true,
			notCheckable = true,
		},
		{
			text = "Notifications",
			checked = function()
				return self.db.profile.notifications.enabled
			end,
			func = function()
				self.db.profile.notifications.enabled = not self.db.profile.notifications.enabled
			end,
		},
		{
			text = "Auto Refresh",
			checked = function()
				return self.db.profile.autoRefresh
			end,
			func = function()
				self.db.profile.autoRefresh = not self.db.profile.autoRefresh
				if self.db.profile.autoRefresh then
					self:ScheduleRepeatingTimer("RefreshRaidList", self.db.profile.refreshInterval)
				else
					self:CancelAllTimers()
				end
			end,
		},
		{
			text = " ",
			isTitle = true,
			notCheckable = true,
		},
		{
			text = "Settings",
			func = function()
				self:OpenConfig()
			end,
			notCheckable = true,
		},
		{
			text = "Close",
			func = function()
				CloseDropDownMenus()
			end,
			notCheckable = true,
		},
	}
	
	local dropdown = CreateFrame("Frame", "WZRaidFinderMinimapMenu", UIParent, "UIDropDownMenuTemplate")
	EasyMenu(menu, dropdown, "cursor", 0, 0, "MENU")
end

function WZRaidFinder:MinimapButton_SetIcon(iconPath)
	if not self.MinimapButton.dataObj then return end
	
	self.MinimapButton.dataObj.icon = iconPath
	LibDBIcon:Refresh("WZRaidFinder", self.db.profile.minimap)
end

function WZRaidFinder:MinimapButton_CreateSimple()
	if self.MinimapButton.simpleButton then
		self.MinimapButton.simpleButton:Show()
		return
	end
	
	local button = CreateFrame("Button", "WZRaidFinderMinimapButton", Minimap)
	button:SetFrameStrata("MEDIUM")
	button:SetWidth(32)
	button:SetHeight(32)
	button:SetFrameLevel(8)
	button:RegisterForClicks("LeftButtonUp", "RightButtonUp")
	button:SetHighlightTexture("Interface\\Minimap\\UI-Minimap-ZoomButton-Highlight")
	
	local icon = button:CreateTexture(nil, "BACKGROUND")
	icon:SetWidth(20)
	icon:SetHeight(20)
	icon:SetPoint("CENTER", 0, 1)
	icon:SetTexture("Interface\\Icons\\Achievement_Boss_LichKing")
	button.icon = icon
	
	local overlay = button:CreateTexture(nil, "OVERLAY")
	overlay:SetWidth(53)
	overlay:SetHeight(53)
	overlay:SetPoint("TOPLEFT")
	overlay:SetTexture("Interface\\Minimap\\MiniMap-TrackingBorder")
	button.overlay = overlay
	
	button:SetPoint("TOPLEFT", Minimap, "TOPLEFT", -15, -15)
	
	button:SetScript("OnClick", function(self, btn)
		WZRaidFinder:OpenMainGUI()
	end)
	
	button:SetScript("OnEnter", function(self)
		GameTooltip:SetOwner(self, "ANCHOR_LEFT")
		GameTooltip:SetText("|cFF16C3F2WZ Raid Finder|r")
		GameTooltip:AddLine("Version: " .. WZRaidFinder.VERSION, 1, 1, 1)
		GameTooltip:AddLine(" ")
		GameTooltip:AddLine("|cFFFFFFFFClick to open|r", 0, 1, 0)
		local count = #WZRaidFinder.raidList
		if count > 0 then
			GameTooltip:AddLine(" ")
			GameTooltip:AddLine(string.format("|cFF00FF00%d raid(s) available|r", count), 1, 1, 1)
		end
		GameTooltip:Show()
	end)
	
	button:SetScript("OnLeave", function(self)
		GameTooltip:Hide()
	end)
	
	button:SetMovable(true)
	button:EnableMouse(true)
	button:RegisterForDrag("LeftButton")
	button:SetScript("OnDragStart", function(self)
		self:StartMoving()
	end)
	button:SetScript("OnDragStop", function(self)
		self:StopMovingOrSizing()
	end)
	
	self.MinimapButton.simpleButton = button
	WZRaidFinder:Print("Minimap button created (simple mode - drag to reposition)")
end

WZRaidFinder.RAID_ICONS = {
	DEFAULT = "Interface\\Icons\\Achievement_Boss_LichKing",
	VANILLA = "Interface\\Icons\\Achievement_Boss_Ragnaros",
	TBC = "Interface\\Icons\\Achievement_Boss_Illidan",
	WOTLK = "Interface\\Icons\\Achievement_Boss_LichKing",
}
