local WZRaidFinder = _G.WZRaidFinder
local AceGUI = LibStub("AceGUI-3.0")

WZRaidFinder.GUI = WZRaidFinder.GUI or {}
local GUI = WZRaidFinder.GUI

local CLASS_ICON_COORDS = {
	["WARRIOR"] = {0, 0.25, 0, 0.25},
	["MAGE"] = {0.25, 0.5, 0, 0.25},
	["ROGUE"] = {0.5, 0.75, 0, 0.25},
	["DRUID"] = {0.75, 1, 0, 0.25},
	["HUNTER"] = {0, 0.25, 0.25, 0.5},
	["SHAMAN"] = {0.25, 0.5, 0.25, 0.5},
	["PRIEST"] = {0.5, 0.75, 0.25, 0.5},
	["WARLOCK"] = {0.75, 1, 0.25, 0.5},
	["PALADIN"] = {0, 0.25, 0.5, 0.75},
	["DEATHKNIGHT"] = {0.25, 0.5, 0.5, 0.75}
}

function WZRaidFinder:OpenMainGUI()
	if not WZRaidFinder.availableInstances or #WZRaidFinder.availableInstances == 0 then
		WZRaidFinder:RequestInstanceList()
		WZRaidFinder:ScheduleTimer(function()
			WZRaidFinder:OpenMainGUI()
		end, 0.2)
		return
	end
	
	if GUI.mainFrame then
		GUI.mainFrame:Show()
		if not WZRaidFinder.lastRaidListUpdate or (time() - WZRaidFinder.lastRaidListUpdate > 10) then
			WZRaidFinder:RequestRaidList()
		end
		return
	end
	
	local frame = AceGUI:Create("Frame")
	frame:SetTitle("WZ Raid Finder")
	frame:SetStatusText("v" .. WZRaidFinder.VERSION)
	frame:SetLayout("Fill")
	frame:SetWidth(750)
	frame:SetHeight(600)
	
	if frame.content then
		frame.content:SetBackdrop({
			bgFile = "Interface\\ChatFrame\\ChatFrameBackground",
			tile = true,
			tileSize = 16,
		})
		frame.content:SetBackdropColor(0.1, 0.1, 0.1, 0.7)
	end
	
	GUI.mainFrame = frame
	
	if frame.frame then
		tinsert(UISpecialFrames, frame.frame:GetName())
	end
	
	frame:SetCallback("OnClose", function(widget)
		widget:Hide()
	end)
	
	local tabGroup = AceGUI:Create("TabGroup")
	tabGroup:SetLayout("Flow")
	tabGroup:SetTabs({
		{text="Raid List", value="list"},
		{text="Create Raid", value="create"},
		{text="My Raid", value="myraid"},
		{text="Settings", value="settings"}
	})
	tabGroup:SetCallback("OnGroupSelected", function(container, event, group)
		container:ReleaseChildren()
		GUI.currentTab = group
		
		if group == "list" then
			GUI.raidListContainer = container
			if not WZRaidFinder.availableInstances or #WZRaidFinder.availableInstances == 0 then
				WZRaidFinder:RequestInstanceList()
			end
			WZRaidFinder:GUI_DrawRaidList(container)
			local currentFilter = WZRaidFinder.db.profile.filters.instance
			WZRaidFinder:RequestRaidList(currentFilter)
		elseif group == "create" then
			GUI.createRaidContainer = container
			if not WZRaidFinder.availableInstances or #WZRaidFinder.availableInstances == 0 then
				WZRaidFinder:RequestInstanceList()
				local loadingLabel = AceGUI:Create("Label")
				loadingLabel:SetText("|cFFFFAA00Loading instance list...\n\nPlease wait a moment and switch to another tab then back.|r")
				loadingLabel:SetFullWidth(true)
				container:AddChild(loadingLabel)
			else
				WZRaidFinder:GUI_DrawCreateRaid(container)
			end
		elseif group == "myraid" then
			GUI.myRaidContainer = container
			WZRaidFinder:GUI_DrawMyRaid(container)
			if not WZRaidFinder.myRaidInfo then
				WZRaidFinder:RequestMyRaid()
			end
		elseif group == "settings" then
			GUI.settingsContainer = container
			WZRaidFinder:GUI_DrawSettings(container)
		end
	end)
	tabGroup:SelectTab("list")
	
	frame:AddChild(tabGroup)
end

function WZRaidFinder:GUI_DrawRaidList(container)
	GUI.raidListContainer = container
	
	local COLOR_GOLD = "|cFFFFD700"
	local COLOR_GREEN = "|cFF00FF00"
	local COLOR_RED = "|cFFFF0000"
	local COLOR_ORANGE = "|cFFFFAA00"
	local COLOR_WHITE = "|cFFFFFFFF"
	local COLOR_RESET = "|r"
	local COLOR_ALLIANCE = "|cFFffd9c0"
	local COLOR_HORDE = "|cFFffd9c0"
	
	local topGroup = AceGUI:Create("SimpleGroup")
	topGroup:SetFullWidth(true)
	topGroup:SetLayout("Flow")
	container:AddChild(topGroup)
	
	local instanceDropdown = AceGUI:Create("Dropdown")
	instanceDropdown:SetLabel("Filter by Instance")
	instanceDropdown:SetWidth(200)
	
	local instanceList = {[""] = "All Instances"}
	local showLegacy = WZRaidFinder.db.profile.filters.showLegacyInstances
	for _, inst in ipairs(WZRaidFinder.availableInstances) do
		if showLegacy or inst.expansion == 2 then
			instanceList[inst.acronym] = inst.name
		end
	end
	instanceDropdown:SetList(instanceList)
	
	local currentFilter = WZRaidFinder.db.profile.filters.instance or ""
	instanceDropdown:SetValue(currentFilter)
	
	if instanceDropdown.text then
		instanceDropdown.text:SetJustifyH("LEFT")
	end
	
	instanceDropdown:SetCallback("OnValueChanged", function(widget, event, key)
		WZRaidFinder.db.profile.filters.instance = (key ~= "" and key or nil)
		WZRaidFinder:RequestRaidList(key ~= "" and key or nil)
		widget:SetValue(key)
	end)
	topGroup:AddChild(instanceDropdown)
	
	GUI.instanceDropdown = instanceDropdown
	
	local gsInput = AceGUI:Create("EditBox")
	gsInput:SetLabel("Min GearScore")
	gsInput:SetWidth(60)
	gsInput:SetText(tostring(WZRaidFinder.db.profile.filters.minGearScore))
	gsInput:SetCallback("OnEnterPressed", function(widget, event, text)
		local gs = tonumber(text) or 0
		WZRaidFinder.db.profile.filters.minGearScore = gs
		GUI.raidListContainer:ReleaseChildren()
		WZRaidFinder:GUI_DrawRaidList(GUI.raidListContainer)
	end)
	topGroup:AddChild(gsInput)
	
	local allianceCheckbox = AceGUI:Create("CheckBox")
	allianceCheckbox:SetLabel("|TInterface\\FriendsFrame\\PlusManz-Alliance:16:16:0:0|t")
	allianceCheckbox:SetWidth(50)
	allianceCheckbox:SetValue(WZRaidFinder.db.profile.filters.showAlliance)
	allianceCheckbox:SetCallback("OnValueChanged", function(widget, event, value)
		WZRaidFinder.db.profile.filters.showAlliance = value
		GUI.raidListContainer:ReleaseChildren()
		WZRaidFinder:GUI_DrawRaidList(GUI.raidListContainer)
	end)
	topGroup:AddChild(allianceCheckbox)
	
	local hordeCheckbox = AceGUI:Create("CheckBox")
	hordeCheckbox:SetLabel("|TInterface\\FriendsFrame\\PlusManz-Horde:16:16:0:0|t")
	hordeCheckbox:SetWidth(50)
	hordeCheckbox:SetValue(WZRaidFinder.db.profile.filters.showHorde)
	hordeCheckbox:SetCallback("OnValueChanged", function(widget, event, value)
		WZRaidFinder.db.profile.filters.showHorde = value
		GUI.raidListContainer:ReleaseChildren()
		WZRaidFinder:GUI_DrawRaidList(GUI.raidListContainer)
	end)
	topGroup:AddChild(hordeCheckbox)
	
	local crossCheckbox = AceGUI:Create("CheckBox")
	crossCheckbox:SetLabel("Cross")
	crossCheckbox:SetWidth(80)
	crossCheckbox:SetValue(WZRaidFinder.db.profile.filters.showCrossFaction)
	crossCheckbox:SetCallback("OnValueChanged", function(widget, event, value)
		WZRaidFinder.db.profile.filters.showCrossFaction = value
		GUI.raidListContainer:ReleaseChildren()
		WZRaidFinder:GUI_DrawRaidList(GUI.raidListContainer)
	end)
	topGroup:AddChild(crossCheckbox)
	
	local fullCheckbox = AceGUI:Create("CheckBox")
	fullCheckbox:SetLabel("Show Full Party")
	fullCheckbox:SetWidth(120)
	fullCheckbox:SetValue(WZRaidFinder.db.profile.filters.showFullRaids)
	fullCheckbox:SetCallback("OnValueChanged", function(widget, event, value)
		WZRaidFinder.db.profile.filters.showFullRaids = value
		GUI.raidListContainer:ReleaseChildren()
		WZRaidFinder:GUI_DrawRaidList(GUI.raidListContainer)
	end)
	topGroup:AddChild(fullCheckbox)
	
	local bottomGroup = AceGUI:Create("SimpleGroup")
	bottomGroup:SetFullWidth(true)
	bottomGroup:SetLayout("Flow")
	container:AddChild(bottomGroup)
	
	local refreshBtn = AceGUI:Create("Button")
	refreshBtn:SetText("Refresh")
	refreshBtn:SetWidth(100)
	refreshBtn:SetCallback("OnClick", function()
		local currentFilter = WZRaidFinder.db.profile.filters.instance
		WZRaidFinder:RequestRaidList(currentFilter)
		WZRaidFinder:Print("Refreshing raid list...")
	end)
	bottomGroup:AddChild(refreshBtn)
	
	local heading = AceGUI:Create("Heading")
	heading:SetFullWidth(true)
	container:AddChild(heading)
	
	local scroll = AceGUI:Create("ScrollFrame")
	scroll:SetFullWidth(true)
	scroll:SetFullHeight(true)
	scroll:SetLayout("Flow")
	container:AddChild(scroll)
	
	GUI.raidListScroll = scroll
	
	local displayedCount = 0
	local minGS = WZRaidFinder.db.profile.filters.minGearScore
	local showCross = WZRaidFinder.db.profile.filters.showCrossFaction
	local showFull = WZRaidFinder.db.profile.filters.showFullRaids
	local showAlliance = WZRaidFinder.db.profile.filters.showAlliance
	local showHorde = WZRaidFinder.db.profile.filters.showHorde
	local MAX_RAIDS_DISPLAY = 50
	
	if #WZRaidFinder.raidList == 0 then
		local noRaidsLabel = AceGUI:Create("Label")
		noRaidsLabel:SetText("|cFFFFFF00No raids available.\n\nClick 'Refresh' to update the list.|r")
		noRaidsLabel:SetFullWidth(true)
		noRaidsLabel:SetFontObject(GameFontNormalLarge)
		scroll:AddChild(noRaidsLabel)
	else
		for i = #WZRaidFinder.raidList, 1, -1 do
			local raid = WZRaidFinder.raidList[i]
			if displayedCount >= MAX_RAIDS_DISPLAY then
				break
			end
			
			local passGSFilter = (raid.minGS >= minGS or minGS == 0)
			local passCrossFilter = (showCross or not raid.isCross)
			local passFullFilter = (showFull or raid.current < raid.max)
			local passFactionFilter = (raid.faction == 0 and showAlliance) or (raid.faction == 1 and showHorde)
			
			if passGSFilter and passCrossFilter and passFullFilter and passFactionFilter then
				displayedCount = displayedCount + 1
				
				local raidGroup = AceGUI:Create("InlineGroup")
				raidGroup:SetFullWidth(true)
				raidGroup:SetLayout("Flow")
				raidGroup:SetTitle("Raid ID : " .. raid.id)
				scroll:AddChild(raidGroup)

				local factionColor = raid.faction == 0 and COLOR_ALLIANCE or COLOR_HORDE
				local modeColor = raid.mode:find("H") and COLOR_RED or COLOR_GREEN
				
				local raceIcon = ""
				if raid.faction == 0 then
					raceIcon = "|TInterface\\FriendsFrame\\PlusManz-Alliance:20:20:0:0|t"
				else
					raceIcon = "|TInterface\\FriendsFrame\\PlusManz-Horde:20:20:0:0|t"
				end
				
				local leaderIcon = ""
				if raid.leaderClass then
					local coords = CLASS_ICON_COORDS[raid.leaderClass]
					if coords then
						leaderIcon = string.format("|TInterface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes:16:16:0:0:256:256:%d:%d:%d:%d|t",
							coords[1]*256, coords[2]*256, coords[3]*256, coords[4]*256)
					end
				end
				
				local cdText = raid.hasCD and (COLOR_RED .. "(Has CD)" .. COLOR_RESET) or (COLOR_GREEN .. "(Fresh)" .. COLOR_RESET)
				local crossText = raid.isCross and (COLOR_ORANGE .. " [Cross-Faction]" .. COLOR_RESET) or ""
				local gsReq = raid.minGS > 0 and string.format("%sMin GS: %d%s", COLOR_GOLD, raid.minGS, COLOR_RESET) or ("|cFF808080No GS Requirement|r")
				
				local instanceLabel = AceGUI:Create("Label")
				instanceLabel:SetText(string.format(
					"%s %s%s%s  •  %s%s%s  •  %s %s%s%s",
					raceIcon, factionColor, raid.map, COLOR_RESET,
					modeColor, raid.mode, COLOR_RESET,
					leaderIcon, COLOR_WHITE, raid.leader, COLOR_RESET
				))
				instanceLabel:SetFullWidth(true)
				if instanceLabel.label then
					instanceLabel.label:SetFont(instanceLabel.label:GetFont(), 13)
				end
				raidGroup:AddChild(instanceLabel)
				
				local detailsLabel = AceGUI:Create("Label")
				detailsLabel:SetText(string.format(
					"%sPlayers:%s %s%d/%d%s  •  %s  •  %s%s",
					COLOR_WHITE, COLOR_RESET, COLOR_WHITE, raid.current, raid.max, COLOR_RESET, cdText, gsReq, crossText
				))
				detailsLabel:SetFullWidth(true)
				raidGroup:AddChild(detailsLabel)
				
				local joinBtn = AceGUI:Create("Button")
				joinBtn:SetText("Join Raid")
				joinBtn:SetWidth(120)
				joinBtn:SetCallback("OnClick", function()
					WZRaidFinder:SendCommand(string.format(".raid join %d", raid.id))
					WZRaidFinder:Print(string.format("Joining raid %d...", raid.id))
				end)
				raidGroup:AddChild(joinBtn)
				
				local whisperBtn = AceGUI:Create("Button")
				whisperBtn:SetText("|TInterface\\GossipFrame\\GossipGossipIcon:20:20:0:-1|t")
				whisperBtn:SetWidth(60)
				whisperBtn:SetCallback("OnClick", function()
					ChatFrame_SendTell(raid.leader)
				end)
				raidGroup:AddChild(whisperBtn)
				
				local infoBtn = AceGUI:Create("Button")
				infoBtn:SetText("|TInterface\\MINIMAP\\Tracking\\None:18:18:0:-1|t")
				infoBtn:SetWidth(60)
				infoBtn:SetCallback("OnClick", function()
					if WZRaidFinder.detailsFrame and WZRaidFinder.currentDetailsRaidId == raid.id then
						local oldFrame = WZRaidFinder.detailsFrame
						WZRaidFinder.detailsFrame = nil
						WZRaidFinder.currentDetailsRaidId = nil
						oldFrame:Hide()
						AceGUI:Release(oldFrame)
					else
						if WZRaidFinder.detailsFrame then
							local oldFrame = WZRaidFinder.detailsFrame
							WZRaidFinder.detailsFrame = nil
							oldFrame:Hide()
							AceGUI:Release(oldFrame)
						end
						WZRaidFinder.currentDetailsRaidId = raid.id
						WZRaidFinder.viewingOwnRaid = false
						WZRaidFinder:RequestRaidDetails(raid.id)
					end
				end)
				raidGroup:AddChild(infoBtn)
			end
		end
	end
	
	local statusLabel = AceGUI:Create("Label")
	statusLabel:SetText(string.format("|cFF00FF00%d raid(s) displayed|r", displayedCount))
	statusLabel:SetFullWidth(true)
	scroll:AddChild(statusLabel)
end

function WZRaidFinder:GUI_DrawCreateRaid(container)
	GUI.createRaidContainer = container
	
	local scroll = AceGUI:Create("ScrollFrame")
	scroll:SetFullWidth(true)
	scroll:SetFullHeight(true)
	scroll:SetLayout("Flow")
	container:AddChild(scroll)
	
	local titleLabel = AceGUI:Create("Label")
	titleLabel:SetText("|cFF16C3F2Create New Raid|r")
	titleLabel:SetFullWidth(true)
	titleLabel:SetFontObject(GameFontNormalLarge)
	scroll:AddChild(titleLabel)
	
	local instanceGroup = AceGUI:Create("InlineGroup")
	instanceGroup:SetTitle("1. Select Raid Instance & Mode")
	instanceGroup:SetFullWidth(true)
	instanceGroup:SetLayout("Flow")
	scroll:AddChild(instanceGroup)
	
	local instanceDropdown = AceGUI:Create("Dropdown")
	instanceDropdown:SetLabel("Raid Instance")
	instanceDropdown:SetRelativeWidth(0.5)
	
	local instanceList = {}
	local showLegacy = WZRaidFinder.db.profile.filters.showLegacyInstances
	for _, inst in ipairs(WZRaidFinder.availableInstances) do
		if inst.expansion == 2 or showLegacy then
			instanceList[inst.acronym] = string.format("%s (%s)", inst.name, inst.acronym)
		end
	end
	instanceDropdown:SetList(instanceList)
	instanceDropdown:SetText("Please select an instance")
	
	if instanceDropdown.text then
		instanceDropdown.text:SetJustifyH("LEFT")
	end
	
	instanceDropdown:SetCallback("OnValueChanged", function(widget, event, key)
		GUI.selectedInstance = key
		GUI.selectedMode = nil
		for _, inst in ipairs(WZRaidFinder.availableInstances) do
			if inst.acronym == key then
				GUI.selectedInstanceData = inst
				if GUI.modeGroup then
					GUI.modeGroup:ReleaseChildren()
					WZRaidFinder:GUI_UpdateModeOptions(GUI.modeGroup, inst.modes)
				end
				break
			end
		end
	end)
	instanceGroup:AddChild(instanceDropdown)
	
	local gsInput = AceGUI:Create("EditBox")
	gsInput:SetLabel("Min GearScore")
	gsInput:SetRelativeWidth(0.25)
	gsInput:SetText("0")
	gsInput:SetCallback("OnEnterPressed", function(widget, event, text)
		GUI.selectedGearScore = tonumber(text) or 0
	end)
	instanceGroup:AddChild(gsInput)
	
	local limitInput = AceGUI:Create("EditBox")
	limitInput:SetLabel("Max Players")
	limitInput:SetRelativeWidth(0.25)
	limitInput:SetText("0")
	limitInput:SetCallback("OnEnterPressed", function(widget, event, text)
		GUI.selectedPlayerLimit = tonumber(text) or 0
	end)
	instanceGroup:AddChild(limitInput)
	
	local modeGroup = AceGUI:Create("SimpleGroup")
	modeGroup:SetFullWidth(true)
	modeGroup:SetLayout("Flow")
	instanceGroup:AddChild(modeGroup)
	
	GUI.modeGroup = modeGroup
	
	local modeLabel = AceGUI:Create("Label")
	modeLabel:SetText("|cFFFFAA00Please select an instance first|r")
	modeLabel:SetFullWidth(true)
	modeGroup:AddChild(modeLabel)
	
	local optionsGroup = AceGUI:Create("InlineGroup")
	optionsGroup:SetTitle("2. Raid Options")
	optionsGroup:SetFullWidth(true)
	optionsGroup:SetLayout("Flow")
	scroll:AddChild(optionsGroup)
	
	local crossCheckbox = AceGUI:Create("CheckBox")
	crossCheckbox:SetLabel("Cross-Faction (Allow both Alliance and Horde)")
	crossCheckbox:SetFullWidth(true)
	crossCheckbox:SetValue(false)
	crossCheckbox:SetCallback("OnValueChanged", function(widget, event, value)
		GUI.selectedCrossFaction = value
	end)
	optionsGroup:AddChild(crossCheckbox)
	
	local silentCheckbox = AceGUI:Create("CheckBox")
	silentCheckbox:SetLabel("Silent Mode (Don't announce in global chat)")
	silentCheckbox:SetFullWidth(true)
	silentCheckbox:SetValue(false)
	silentCheckbox:SetCallback("OnValueChanged", function(widget, event, value)
		GUI.selectedSilentMode = value
	end)
	optionsGroup:AddChild(silentCheckbox)
	
	local heading = AceGUI:Create("Heading")
	heading:SetFullWidth(true)
	scroll:AddChild(heading)
	
	local createBtn = AceGUI:Create("Button")
	createBtn:SetText("Create Raid")
	createBtn:SetFullWidth(true)
	createBtn:SetCallback("OnClick", function()
		if not GUI.selectedInstance or not GUI.selectedMode then
			WZRaidFinder:Print("|cFFFF0000Please select both instance and mode!|r")
			return
		end
		
		local cmd = ".raid create"
		if GUI.selectedCrossFaction and GUI.selectedSilentMode then
			cmd = cmd .. " cross silent"
		elseif GUI.selectedCrossFaction then
			cmd = cmd .. " cross"
		elseif GUI.selectedSilentMode then
			cmd = cmd .. " silent"
		end
		
		cmd = cmd .. " " .. GUI.selectedInstance .. " " .. GUI.selectedMode
		
		if GUI.selectedGearScore and GUI.selectedGearScore > 0 then
			cmd = cmd .. " " .. GUI.selectedGearScore
		end
		
		if GUI.selectedPlayerLimit and GUI.selectedPlayerLimit > 0 then
			cmd = cmd .. " " .. GUI.selectedPlayerLimit
		end
		
		WZRaidFinder:SendCommand(cmd)
		WZRaidFinder:Print("Creating raid...")
		
		WZRaidFinder:ScheduleTimer(function()
			WZRaidFinder:RequestMyRaid()
			if GUI.mainFrame and GUI.mainFrame.frame:IsShown() then
				local tabGroup = GUI.mainFrame.children[1]
				if tabGroup and tabGroup.SelectTab then
					tabGroup:SelectTab("myraid")
				end
			end
		end, 1.5)
	end)
	scroll:AddChild(createBtn)
end

function WZRaidFinder:GUI_UpdateModeOptions(container, modes)
	container:ReleaseChildren()
	
	local modeLabel = AceGUI:Create("Label")
	modeLabel:SetText("Select raid mode:")
	modeLabel:SetFullWidth(true)
	container:AddChild(modeLabel)
	
	local modeButtons = {
		{key = 1, name = "10N", label = "10 Normal"},
		{key = 2, name = "25N", label = "25 Normal"},
		{key = 4, name = "10H", label = "10 Heroic"},
		{key = 8, name = "25H", label = "25 Heroic"},
		{key = 16, name = "40", label = "40 Players"},
	}
	
	if not GUI.modeButtonWidgets then
		GUI.modeButtonWidgets = {}
	end
	GUI.modeButtonWidgets = {}
	
	if GUI.selectedMode then
		for _, modeBtn in ipairs(modeButtons) do
			if bit.band(modes, modeBtn.key) > 0 and modeBtn.name == GUI.selectedMode then
				local btn = AceGUI:Create("Button")
				btn:SetText("|cFF00FF00" .. modeBtn.label .. "|r")
				btn:SetWidth(150)
				btn:SetCallback("OnClick", function()
					GUI.selectedMode = nil
					WZRaidFinder:Print("Mode deselected. Choose again.")
					WZRaidFinder:GUI_UpdateModeOptions(container, modes)
				end)
				container:AddChild(btn)
				break
			end
		end
	else
		for _, modeBtn in ipairs(modeButtons) do
			if bit.band(modes, modeBtn.key) > 0 then
				local btn = AceGUI:Create("Button")
				btn:SetText("|cFFFFAA00" .. modeBtn.label .. "|r")
				btn:SetWidth(150)
				btn:SetCallback("OnClick", function()
					GUI.selectedMode = modeBtn.name
					WZRaidFinder:Print(string.format("Selected mode: %s", modeBtn.name))
					WZRaidFinder:GUI_UpdateModeOptions(container, modes)
				end)
				container:AddChild(btn)
				
				table.insert(GUI.modeButtonWidgets, btn)
			end
		end
	end
	
	container:DoLayout()
	if GUI.createRaidContainer then
		GUI.createRaidContainer:DoLayout()
	end
end

function WZRaidFinder:GUI_DrawMyRaid(container)
	GUI.myRaidContainer = container
	
	local scroll = AceGUI:Create("ScrollFrame")
	scroll:SetFullWidth(true)
	scroll:SetFullHeight(true)
	scroll:SetLayout("Flow")
	container:AddChild(scroll)
	
	local refreshBtn = AceGUI:Create("Button")
	refreshBtn:SetText("Refresh My Raid")
	refreshBtn:SetWidth(150)
	refreshBtn:SetCallback("OnClick", function()
		WZRaidFinder:RequestMyRaid()
		WZRaidFinder:Print("Refreshing...")
	end)
	scroll:AddChild(refreshBtn)
	
	local heading = AceGUI:Create("Heading")
	heading:SetFullWidth(true)
	scroll:AddChild(heading)
	
	if WZRaidFinder.myRaidInfo then
		local raid = WZRaidFinder.myRaidInfo
		
		local infoGroup = AceGUI:Create("InlineGroup")
		infoGroup:SetTitle("Your Active Raid")
		infoGroup:SetFullWidth(true)
		infoGroup:SetLayout("Flow")
		scroll:AddChild(infoGroup)
		
		local detailsLabel = AceGUI:Create("Label")
		local modeColor = raid.mode:find("H") and "|cFFFF0000" or "|cFF00FF00"
		local crossText = raid.isCross and "|cFFFFAA00[Cross-Faction]|r" or ""
		local gsText = raid.minGS > 0 and string.format("Min GS: %d", raid.minGS) or "No GS requirement"
		
		detailsLabel:SetText(string.format(
			"|cFFFFFFFFRaid ID:|r |cFFFFAA00#%d|r\n" ..
			"|cFFFFFFFFRaid:|r %s\n" ..
			"|cFFFFFFFFMode:|r %s%s|r\n" ..
			"|cFFFFFFFFPlayers:|r %d/%d\n" ..
			"|cFFFFFFFFGearScore:|r %s\n" ..
			"%s",
			raid.id,
			raid.map, modeColor, raid.mode, 
			raid.current, raid.max,
			gsText, crossText
		))
		detailsLabel:SetFullWidth(true)
		infoGroup:AddChild(detailsLabel)
		
		local detailsBtn = AceGUI:Create("Button")
		detailsBtn:SetText("View Details")
		detailsBtn:SetWidth(150)
		detailsBtn:SetCallback("OnClick", function()
			if raid.id then
				if WZRaidFinder.detailsFrame and WZRaidFinder.currentDetailsRaidId == raid.id then
					local oldFrame = WZRaidFinder.detailsFrame
					WZRaidFinder.detailsFrame = nil
					WZRaidFinder.currentDetailsRaidId = nil
					oldFrame:Hide()
					AceGUI:Release(oldFrame)
				else
					if WZRaidFinder.detailsFrame then
						local oldFrame = WZRaidFinder.detailsFrame
						WZRaidFinder.detailsFrame = nil
						oldFrame:Hide()
						AceGUI:Release(oldFrame)
					end
					WZRaidFinder.currentDetailsRaidId = raid.id
					WZRaidFinder.viewingOwnRaid = true
					WZRaidFinder:RequestRaidDetails(raid.id)
				end
			end
		end)
		infoGroup:AddChild(detailsBtn)
		
		local cancelBtn = AceGUI:Create("Button")
		cancelBtn:SetText("Cancel Raid")
		cancelBtn:SetWidth(150)
		cancelBtn:SetCallback("OnClick", function()
			WZRaidFinder:SendCommand(".raid cancel")
			WZRaidFinder:Print("Cancelling raid...")
			WZRaidFinder.myRaidInfo = nil
			if GUI.myRaidContainer then
				GUI.myRaidContainer:ReleaseChildren()
				WZRaidFinder:GUI_DrawMyRaid(GUI.myRaidContainer)
			end
			WZRaidFinder:ScheduleTimer(function()
				WZRaidFinder:RequestMyRaid()
			end, 0.5)
		end)
		infoGroup:AddChild(cancelBtn)
	else
		local noRaidLabel = AceGUI:Create("Label")
		noRaidLabel:SetText("|cFFFFAA00You don't have an active raid.\n\nCreate one from the 'Create Raid' tab!|r")
		noRaidLabel:SetFullWidth(true)
		noRaidLabel:SetFontObject(GameFontNormalLarge)
		scroll:AddChild(noRaidLabel)
	end
end

function WZRaidFinder:GUI_DrawSettings(container)
	local scroll = AceGUI:Create("ScrollFrame")
	scroll:SetFullWidth(true)
	scroll:SetFullHeight(true)
	scroll:SetLayout("Flow")
	container:AddChild(scroll)
	
	local chatCheckbox = AceGUI:Create("CheckBox")
	chatCheckbox:SetLabel("Show chat messages")
	chatCheckbox:SetFullWidth(true)
	chatCheckbox:SetValue(WZRaidFinder.db.profile.chat.showMessages)
	chatCheckbox:SetCallback("OnValueChanged", function(widget, event, value)
		WZRaidFinder.db.profile.chat.showMessages = value
	end)
	scroll:AddChild(chatCheckbox)
	
	local legacyCheckbox = AceGUI:Create("CheckBox")
	legacyCheckbox:SetLabel("Show Classic/TBC raid instances")
	legacyCheckbox:SetFullWidth(true)
	legacyCheckbox:SetValue(WZRaidFinder.db.profile.filters.showLegacyInstances)
	legacyCheckbox:SetCallback("OnValueChanged", function(widget, event, value)
		WZRaidFinder.db.profile.filters.showLegacyInstances = value
		WZRaidFinder:Print("Setting saved. Instance list will update when you switch tabs.")
	end)
	scroll:AddChild(legacyCheckbox)
end

function WZRaidFinder:GUI_ShowRaidDetails(details, isOwnRaid)
	if not details then return end
	
	
	local frame = AceGUI:Create("Frame")
	frame:SetTitle("Raid Details - " .. details.map)
	frame:SetWidth(500)
	frame:SetHeight(450)
	frame:SetLayout("Flow")
	frame:SetCallback("OnClose", function(widget)
		if WZRaidFinder.detailsFrame == widget then
			WZRaidFinder.detailsFrame = nil
			WZRaidFinder.currentDetailsRaidId = nil
		end
		AceGUI:Release(widget)
	end)
	
	self.detailsFrame = frame
	self.currentDetailsRaidId = details.id
	
	if frame.content then
		frame.content:SetBackdrop({
			bgFile = "Interface\\ChatFrame\\ChatFrameBackground",
			tile = true,
			tileSize = 16,
		})
		frame.content:SetBackdropColor(0.15, 0.15, 0.15, 1)
	end
	
	local scroll = AceGUI:Create("ScrollFrame")
	scroll:SetFullWidth(true)
	scroll:SetFullHeight(true)
	scroll:SetLayout("Flow")
	frame:AddChild(scroll)
	
	if scroll.content then
		scroll.content:SetBackdrop({
			bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
			tile = true,
			tileSize = 16,
		})
		scroll.content:SetBackdropColor(0.15, 0.15, 0.15, 1)
	end
	
	local headerGroup = AceGUI:Create("InlineGroup")
	headerGroup:SetTitle("Raid Information")
	headerGroup:SetFullWidth(true)
	headerGroup:SetLayout("Flow")
	scroll:AddChild(headerGroup)
	
	local instanceLabel = AceGUI:Create("Label")
	instanceLabel:SetText(string.format("|cFFFFFFFFInstance:|r %s", details.map))
	instanceLabel:SetFullWidth(true)
	headerGroup:AddChild(instanceLabel)
	
	if details.mode and details.mode ~= "" then
		local modeLabel = AceGUI:Create("Label")
		local modeColor = details.mode:find("H") and "|cFFFF0000" or "|cFF00FF00"
		modeLabel:SetText(string.format("|cFFFFFFFFMode:|r %s%s|r", modeColor, details.mode))
		modeLabel:SetFullWidth(true)
		headerGroup:AddChild(modeLabel)
	end
	
	local leaderLabel = AceGUI:Create("Label")
	leaderLabel:SetText(string.format("|cFFFFFFFFLeader:|r |cFFFFAA00%s|r", details.leader))
	leaderLabel:SetFullWidth(true)
	headerGroup:AddChild(leaderLabel)
	
	local gsLabel = AceGUI:Create("Label")
	if details.minGS and details.minGS > 0 then
		gsLabel:SetText(string.format("|cFFFFD700⚔ Min GearScore:|r |cFFFFFFFF%d|r", details.minGS))
	else
		gsLabel:SetText("|cFF00FF00 No GearScore Requirement|r")
	end
	gsLabel:SetFullWidth(true)
	headerGroup:AddChild(gsLabel)
	
	if details.isCross then
		local crossLabel = AceGUI:Create("Label")
		crossLabel:SetText("|cFFFFAA00Cross-Faction Allowed|r")
		crossLabel:SetFullWidth(true)
		headerGroup:AddChild(crossLabel)
	end
	
	if details.members and #details.members > 0 then
		local membersGroup = AceGUI:Create("InlineGroup")
		membersGroup:SetTitle(string.format("Group Members (%d/%d)", #details.members, details.maxMembers or 25))
		membersGroup:SetFullWidth(true)
		membersGroup:SetLayout("Flow")
		scroll:AddChild(membersGroup)
		
		local classColors = {
			[1] = "|cFFC79C6E",
			[2] = "|cFFF58CBA",
			[3] = "|cFFABD473",
			[4] = "|cFFFFF569",
			[5] = "|cFFFFFFFF",
			[6] = "|cFFC41F3B",
			[7] = "|cFF0070DE",
			[8] = "|cFF40C7EB",
			[9] = "|cFF8788EE",
			[11] = "|cFFFF7D0A",
		}
		
		for i, member in ipairs(details.members) do
			local memberLabel = AceGUI:Create("Label")
			local classColor = classColors[member.class] or "|cFFFFFFFF"
			memberLabel:SetText(string.format(
				"%d. %s%s|r - |cFFFFFFFFLevel %d|r - |cFFFFAA00GS: %d|r",
				i, classColor, member.name, member.level, member.gearscore
			))
			memberLabel:SetFullWidth(true)
			membersGroup:AddChild(memberLabel)
		end
	end
	
	local btnGroup = AceGUI:Create("SimpleGroup")
	btnGroup:SetFullWidth(true)
	btnGroup:SetLayout("Flow")
	scroll:AddChild(btnGroup)
	
	if not WZRaidFinder.viewingOwnRaid then
		local joinBtn = AceGUI:Create("Button")
		joinBtn:SetText("Join This Raid")
		joinBtn:SetWidth(150)
		joinBtn:SetCallback("OnClick", function()
			WZRaidFinder:SendCommand(string.format(".raid join %d", details.id))
			WZRaidFinder:Print(string.format("Joining raid %d...", details.id))
			frame:Hide()
		end)
		btnGroup:AddChild(joinBtn)
		
		local whisperBtn = AceGUI:Create("Button")
		whisperBtn:SetText("|TInterface\\GossipFrame\\GossipGossipIcon:20:20:0:-1|t")
		whisperBtn:SetWidth(60)
		whisperBtn:SetCallback("OnClick", function()
			ChatFrame_SendTell(details.leader)
		end)
		btnGroup:AddChild(whisperBtn)
	end
	
	local closeBtn = AceGUI:Create("Button")
	closeBtn:SetText("Close")
	closeBtn:SetWidth(100)
	closeBtn:SetCallback("OnClick", function()
		frame:Hide()
	end)
	btnGroup:AddChild(closeBtn)
end

